/**
 * \file: mspin_udp_broadcast.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN UDP Broadcast Support
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
* \copyright: (c) 2016 Bosch SoftTec GmbH
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_udp_broadcast.h"
#include "mspin_udp_helper.h"
#include "mspin_logging.h"

#include <errno.h>          //errno
#include <arpa/inet.h>      //inet_ntop
#include <sys/prctl.h>      //prctl, PR_SET_NAME

//#define MSPIN_UDP_MESSAGE_OLD_FORMAT

typedef struct
{
    mspin_udp_MessageParameter_t message;
    in_addr_t ipAddr;
    in_addr_t netmask;
    BOOL quitUDPBroadcasting;
    pthread_t broadcastThreadID;
    MSPIN_OnUDPBroadcastEnd onUDPBroadcastEnd;
    void* onUDPBroadcastContext;
    pthread_mutex_t performBroadcastLock;
    pthread_cond_t abortBroadcast;
} mspin_udp_BroadcastContext_t;

mspin_udp_BroadcastContext_t *gpUDPBroadcastContext = NULL;

static void mspin_udp_freeBroadcastContext(mspin_udp_BroadcastContext_t** ppContext)
{
    //Frees only memory. Does not destroy any pthread_* variables
    mspin_udp_BroadcastContext_t* pContext = *ppContext;

    if (pContext)
    {
        mspin_udp_freeBTName(&(pContext->message));

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) deleting context...",
                __FUNCTION__, pContext);

        free(pContext);
        *ppContext = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) context deleted",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(context=%p) WARNING: UDP broadcast context is NULL, nothing to do",
                __FUNCTION__, pContext);
    }
}

static mspin_udp_BroadcastContext_t* mspin_udp_createBroadcastContext(void)
{
    mspin_udp_BroadcastContext_t* pContext = NULL;
    int rc = -1;

    pContext = malloc(sizeof(mspin_udp_BroadcastContext_t));

    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate memory for broadcast context",
                __FUNCTION__);
        return NULL;
    }

    memset(pContext, 0, sizeof(mspin_udp_BroadcastContext_t));

    //Initialize broadcast mutex
    rc = pthread_mutex_init(&(pContext->performBroadcastLock), NULL); //use default mutex attributes
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize broadcast mutex with %d",
                __FUNCTION__, rc);

        mspin_udp_freeBroadcastContext(&pContext); //deletes only memory
        return NULL;
    }

    //Initialize broadcast condition variable
    rc = pthread_cond_init(&(pContext->abortBroadcast), NULL); //use default condition attributes
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize broadcast mutex with %d",
                __FUNCTION__, rc);

        pthread_mutex_destroy(&(pContext->performBroadcastLock));
        mspin_udp_freeBroadcastContext(&pContext); //deletes only memory
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s() UDP broadcast context=%p created",
            __FUNCTION__, pContext);

    return pContext;
}

static void mspin_udp_deleteBroadcastContext(mspin_udp_BroadcastContext_t** ppContext)
{
    //Destroys the pthread_* variables and then frees the memory
    mspin_udp_BroadcastContext_t* pContext = *ppContext;
    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered", __FUNCTION__, pContext);

        pthread_mutex_destroy(&(pContext->performBroadcastLock));
        pthread_cond_destroy(&(pContext->abortBroadcast));

        mspin_udp_freeBroadcastContext(ppContext);

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) finished", __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(*ctx=%p) FATAL ERROR: pContext is NULL",
                __FUNCTION__, ppContext);
    }
}

static void* mspin_udp_broadcastThread(void* exinf)
{
    int sockfd = -1;
    int broadcastOption = 1;
    in_addr_t broadcast = 0;
    struct sockaddr_in broadcastAddr = {0};
    char hostString[INET_ADDRSTRLEN] = "";
    char maskString[INET_ADDRSTRLEN] = "";
    char broadcastString[INET_ADDRSTRLEN] = "";
    char udpMessage[MSPIN_UDP_PACKET_MAX_LENGTH] = "";
    struct timespec ts = {0};
    int errorCount = 0;
    MSPIN_UDP_BROADCAST_END_REASON endReason = MSPIN_UDP_BROADCAST_END_REASON_QUIT;

    //Check context
    mspin_udp_BroadcastContext_t *pContext = (mspin_udp_BroadcastContext_t*)exinf;
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(exinf=%p) FATAL ERROR: Context is NULL",
                __FUNCTION__, exinf);

        //Callback can't be issued because context is not valid

        pthread_exit(NULL);
        return NULL;
    }

    //Set thread name
    //                  123456789 123456
    prctl(PR_SET_NAME, "mspin_udpBroadc", 0, 0, 0);

    //Generate broadcast address from IP and mask
    broadcast = pContext->ipAddr | ~pContext->netmask;

    //Debug print broadcast address
    inet_ntop(AF_INET, &pContext->ipAddr, hostString, INET_ADDRSTRLEN);
    inet_ntop(AF_INET, &pContext->netmask, maskString, INET_ADDRSTRLEN);
    inet_ntop(AF_INET, &broadcast, broadcastString, INET_ADDRSTRLEN);

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) using broadcast addr='%s' (ip='%s', mask='%s')",
            __FUNCTION__, exinf, *broadcastString ? broadcastString : "n/a", *hostString ? hostString : "n/a",
            *maskString ? maskString : "n/a");

    //Create socket
    sockfd = socket(AF_INET, SOCK_DGRAM, 0);
    if (-1 == sockfd)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: Failed to create socket with '%s'(%d)",
                __FUNCTION__, exinf, strerror(errno), errno);

        close(sockfd);

        if (pContext->onUDPBroadcastEnd)
        {
            pContext->onUDPBroadcastEnd(MSPIN_UDP_BROADCAST_END_REASON_SETUP_FAILED, pContext->onUDPBroadcastContext);
        }

        pthread_exit(exinf);
        return NULL;
    }

    //Set socket options for broadcasting
    if (-1 == setsockopt(sockfd, SOL_SOCKET, SO_BROADCAST, &broadcastOption, sizeof broadcastOption))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: Failed set socket options for broadcasting with %s'(%d)",
                __FUNCTION__, exinf, strerror(errno), errno);

        close(sockfd);

        if (pContext->onUDPBroadcastEnd)
        {
            pContext->onUDPBroadcastEnd(MSPIN_UDP_BROADCAST_END_REASON_SETUP_FAILED, pContext->onUDPBroadcastContext);
        }

        pthread_exit(exinf);
        return NULL;
    }

    //Construct local address structure
    memset(&broadcastAddr, 0, sizeof(broadcastAddr));
    broadcastAddr.sin_family = AF_INET;
    broadcastAddr.sin_port = htons(pContext->message.udpPort);  // short, network byte order
    broadcastAddr.sin_addr.s_addr = broadcast;

    //Create UDP message
    if (*hostString)
    {
        if (MSPIN_SUCCESS != mspin_udp_createUDPMessage(&(pContext->message), udpMessage, sizeof(udpMessage), hostString))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: UDP message too long -> abort",
                    __FUNCTION__, exinf);

            close(sockfd);

            if (pContext->onUDPBroadcastEnd)
            {
                pContext->onUDPBroadcastEnd(MSPIN_UDP_BROADCAST_END_REASON_IP_ADDRESS_FAILURE,
                        pContext->onUDPBroadcastContext);
            }

            pthread_exit(exinf);
            return NULL;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) message to be sent is: '%s'",
                    __FUNCTION__, exinf, udpMessage);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: host address is empty",
                __FUNCTION__, exinf);

        close(sockfd);

        if (pContext->onUDPBroadcastEnd)
        {
            pContext->onUDPBroadcastEnd(MSPIN_UDP_BROADCAST_END_REASON_IP_ADDRESS_FAILURE,
                    pContext->onUDPBroadcastContext);
        }

        pthread_exit(exinf);
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) start UDP broadcasting using addr='%s'",
            __FUNCTION__, exinf, broadcastString);

    pthread_mutex_lock(&(pContext->performBroadcastLock));


    /* PRQA: Lint Message 64: accept type mismatch */
    /*lint -save -e64*/
    //Start broadcasting
    while(!pContext->quitUDPBroadcasting)
    {
        if (-1 == sendto(sockfd, udpMessage, strlen(udpMessage), 0, (struct sockaddr *)&broadcastAddr, sizeof(broadcastAddr)))
        {
            if (++errorCount >= 3)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(exinf=%p) ERROR: Failed %d to send UDP broadcast message with '%s'(%d)",
                        __FUNCTION__, exinf, errorCount, strerror(errno), errno);

                endReason = MSPIN_UDP_BROADCAST_END_REASON_SEND_FAILED;
                pContext->quitUDPBroadcasting = TRUE;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityWarn,
                        "%s(exinf=%p) WARNING: Failed %d to send UDP broadcast message with '%s'(%d)",
                        __FUNCTION__, exinf, errorCount, strerror(errno), errno);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) UDP packet sent with mesg='%s'",
                    __FUNCTION__, exinf, udpMessage);
        }

        //Wait till timeout or end condition. First set timer
        clock_gettime(CLOCK_REALTIME, &ts);
        ts.tv_sec += pContext->message.timeout/1000;
        ts.tv_nsec += (pContext->message.timeout % 1000) * 1000000;
        if (ts.tv_nsec >= 1000000000L)
        {
            ts.tv_sec++;
            ts.tv_nsec -= 1000000000L;
        }

        pthread_cond_timedwait(&(pContext->abortBroadcast), &(pContext->performBroadcastLock), &ts);
    }
    /*lint -restore*/

    pthread_mutex_unlock(&(pContext->performBroadcastLock));

    //Close socket
    close(sockfd);

    if (pContext->onUDPBroadcastEnd)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) issue UDP broadcast end CB", __FUNCTION__, exinf);
        pContext->onUDPBroadcastEnd(endReason, pContext->onUDPBroadcastContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(exinf=%p) WARNING: No UDP broadcast end callback registered",
                __FUNCTION__, exinf);
    }

    // if not stopped from outside, we have to clean up
    if (endReason == MSPIN_UDP_BROADCAST_END_REASON_SEND_FAILED)
    {
        mspin_udp_deleteBroadcastContext(&gpUDPBroadcastContext);
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) terminate thread", __FUNCTION__, exinf);

    pthread_exit(exinf);
    return NULL;
}

static S32 mspin_udp_startBroadcastingThread(mspin_udp_BroadcastContext_t* pContext)
{
    pthread_attr_t attr;
    S32 rc = -1;

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return rc;
    }

    pContext->quitUDPBroadcasting = FALSE;

    rc = pthread_create(&pContext->broadcastThreadID, &attr, mspin_udp_broadcastThread, (void*)pContext);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, pContext->broadcastThreadID);
    }

    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Setting name failed with rc=%d", __FUNCTION__, rc);
        pContext->broadcastThreadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

static void mspin_udp_stopUDPBroadcastThread(mspin_udp_BroadcastContext_t* pContext)
{
    void* status = NULL;

    if (pContext && (0 != pContext->broadcastThreadID))
    {
        pthread_t threadID = pContext->broadcastThreadID;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) quit and wake up thread=%lu",
                __FUNCTION__, pContext, threadID);

        pthread_mutex_lock(&(pContext->performBroadcastLock));
        pContext->quitUDPBroadcasting = true;
        pthread_cond_signal(&(pContext->abortBroadcast));
        pthread_mutex_unlock(&(pContext->performBroadcastLock));

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) join thread=%lu", __FUNCTION__, pContext, threadID);

        (void)pthread_join(pContext->broadcastThreadID, &status);
        pContext->broadcastThreadID = 0;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) thread=%lu joined", __FUNCTION__, pContext, threadID);
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p) FATAL ERROR: Context handle is NULL",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(ctx=%p) WARNING: Broadcast thread ID is 0",
                __FUNCTION__, pContext);
    }
}

MSPIN_ERROR mspin_udp_startUDPBroadcasting(S32 udpPort, const U8* interface,
        MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters, U32 timeout,
        MSPIN_OnUDPBroadcastEnd onUDPBroadcastEnd, void* onUDPBroadcastEndContext)
{
    MSPIN_ERROR result = mspin_udp_validateMessageParameters(&messageParameters);
    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(udpPort=%d, interface='%s') ERROR: Failed to validate message parameters with %s",
               __FUNCTION__, udpPort, interface, MSPIN_GetErrorName(result));
        return result;
    }

    //Initialize server context
    if (!gpUDPBroadcastContext)
    {
        gpUDPBroadcastContext = mspin_udp_createBroadcastContext();

        if (!gpUDPBroadcastContext)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(udpPort=%d, interface='%s') ERROR: Failed to create broadcast context",
                   __FUNCTION__, udpPort, interface);
            return MSPIN_ERROR_GENERAL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(udpPort=%d, interface='%s') WARNING: UDP broadcast context is already present -> return ALREADY_RUNNING",
                __FUNCTION__, udpPort, interface);
        return MSPIN_ERROR_ALREADY_RUNNING;
    }

    //Store and reset parameters
    gpUDPBroadcastContext->onUDPBroadcastEnd = onUDPBroadcastEnd;
    gpUDPBroadcastContext->onUDPBroadcastContext = onUDPBroadcastEndContext;
    gpUDPBroadcastContext->quitUDPBroadcasting = FALSE;
    gpUDPBroadcastContext->message.udpPort = udpPort;
    gpUDPBroadcastContext->message.timeout = timeout;

    //Copy message parameters
    result = mspin_udp_copyMessageParameters(&messageParameters, &(gpUDPBroadcastContext->message));
    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(udpPort=%d, interface='%s') ERROR: Failed to copy message parameters -> delete context again",
                __FUNCTION__, udpPort, interface);
        mspin_udp_deleteBroadcastContext(&gpUDPBroadcastContext);
    }

    if (MSPIN_SUCCESS == result)
    {
        //Get local IP and broadcast address for specified interface
        result = mspin_udp_getBroadcastAddress(interface, &(gpUDPBroadcastContext->ipAddr), &(gpUDPBroadcastContext->netmask));
        if (MSPIN_SUCCESS == result)
        {
            //Start UDP broadcasting thread
            if (0 != mspin_udp_startBroadcastingThread(gpUDPBroadcastContext))
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(udpPort=%d, interface='%s') ERROR: Failed to start broadcasting thread -> delete context again",
                        __FUNCTION__, udpPort, interface);

                mspin_udp_deleteBroadcastContext(&gpUDPBroadcastContext);

                result = MSPIN_ERROR_GENERAL;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(udpPort=%d, interface='%s') ERROR: Failed to get broadcast address -> delete context again",
                    __FUNCTION__, udpPort, interface);

            mspin_udp_deleteBroadcastContext(&gpUDPBroadcastContext);
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(udpPort=%d, interface='%s') returns '%s'",
            __FUNCTION__, udpPort, interface, MSPIN_GetErrorName(result));

    return result;
}

MSPIN_ERROR mspin_udp_stopUDPBroadcasting(void)
{
    if (gpUDPBroadcastContext)
    {
        //Check if called from UDP broadcast thread
        if (gpUDPBroadcastContext->broadcastThreadID == pthread_self())
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Not allowed to call from broadcast thread! For example when issued from CB 'onUDPBroadcastEnd'",
                    __FUNCTION__);
            return MSPIN_ERROR_NOT_ALLOWED;
        }

        mspin_udp_stopUDPBroadcastThread(gpUDPBroadcastContext);

        //Reset values
        gpUDPBroadcastContext->message.udpPort = 0;
        gpUDPBroadcastContext->message.tcpPort = 0;
        gpUDPBroadcastContext->message.timeout = 0;
        gpUDPBroadcastContext->onUDPBroadcastEnd = NULL;
        gpUDPBroadcastContext->onUDPBroadcastContext = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s() thread joined => delete context", __FUNCTION__);

        mspin_udp_deleteBroadcastContext(&gpUDPBroadcastContext);

        mspin_log_printLn(eMspinVerbosityDebug, "%s() done", __FUNCTION__);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: No UDP broadcast context found", __FUNCTION__);
    }

    return MSPIN_SUCCESS;
}
